/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "lx_net.h"

/*yes for some reason this function duplicates lx_map_file_print_node
  almost line for line.*/

static int
lx_print_node (lx_t*lx, lx_node_t*n)
{
  int i, c = 1;
  lx_node_t*nn;
  
  insist (lx && n);
  
  if (n->xbar)
  {
    for (c = 0, i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
      c += lx_connected (n, i) ? 1 : 0;
  }
  else
  {
    for (c = 0, i = 0; i < LX_HOST_SIZE; i++)
      c += lx_connected (n, i) ? 1 : 0;
  }

  mi_println_map (lx, "%s - " lx_node_format, n->xbar ? "s" : "h", lx_node_args (n));
  mi_println_map (lx, "%d", c);

  for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
  {
    if ((nn = lx_get_node (n, i)))
    {
      insist (i >= n->first);
      insist (lx_get_port (n, i) >= nn->first);
      mi_println_map (lx, "%d %s - " lx_node_format " %d", i - n->first, nn->xbar ? "s" : "h", lx_node_args (nn), lx_get_port (n, i) - nn->first);
    }
  }
  
  if (n->xbar)
  {
    unsigned id = lx_xbar_c (n)->id;
    mi_println_map (lx, "ID %u", id);
    mi_println_map (lx, "decodeID %u:%u", id / 16, id % 16);
  }
  else
  {
    mi_println_map (lx, "address " lx_mac_format, lx_mac_args (lx_host_c (n)->mac_address));
    mi_println_map (lx, "hostType %d", lx_host_c (n)->host_type);
  }
  mi_println_map (lx, "");
  
  return 1;  
  except: return 0;
}

static int
lx_print_nodes (lx_t*lx, lx_queue_t*queue)
{
  lx_node_t*n;
  
  insist (lx && queue);

  for (n = queue->head; n; n = mi_index2node (n->next))
    if (!lx_print_node (lx, n))
      return 0;
  
  return 1;
  except: return 0;
  
}

int
lx_map_print (lx_t*lx, lx_map_t*m)
{
  insist (lx && m);
  mi_println_map (lx, "; map_address " lx_mac_format " map_version %u", lx_mac_args (m->map_address), m->map_version);
  
  return lx_print_nodes (lx, &m->hosts) && lx_print_nodes (lx, &m->xbars);
  except: return 0;
}


int
lx_receive (lx_t*lx, int*iport, char*p, unsigned length, int early)
{
  int reason, iiport;
  unsigned i, duration;
  void*contexts [LX_NUM_SMALLS + LX_NUM_REPLIES + LX_NUM_BIGS];
  unsigned message_length;
  
  insist (lx->pendings [LX_BIG_SEND] || lx->pendings [LX_SMALL_SEND] || lx->pendings [LX_REPLY_SEND] || lx->duration || early);
  insist (!lx->in_receive);

  lx->in_receive = 1;  
  
  if (lx->set_alarm)
  {
    insist (lx->duration);
    lx->set_alarm = 0;
    mi_set_alarm (lx, lx->duration);
  }  
  
  duration = lx->duration;
  
  reason = mi_wait (lx, &iiport, (char*) lx->receive_buffer, &message_length, contexts, early);  
  
  switch (reason)
  {
    case LX_RECEIVE:

      if (((lx_header_t*) lx->receive_buffer)->version != LX_VERSION && mi_htons (((lx_header_t*)lx->receive_buffer)->subtype) != LX_XBAR32)
	break;
      
      lx_map_receive_callback (lx, iiport, (lx_packet_t*) lx->receive_buffer, message_length);
      
      insist (duration == lx->duration);
      
      if (message_length >= length && p)
      {
	insist (iport);
	
	if (p != lx->receive_buffer)
	  mi_memcpy (p, (char*) lx->receive_buffer, length);
	lx->in_receive = 0;
	*iport = iiport;
	return message_length;
      }
      /*mi_c (("dropped message length %d", message_length));*/
      
      break;
    case LX_ALARM:
      insist (lx->duration);
      lx->duration = 0;
      /*mi_c (("alarm time %u", mi_rtc - rtc));*/
      
      if (lx->timeout > LX_TIMEOUT_DECREASE)
      {
	lx->timeout -= LX_TIMEOUT_DECREASE;
	if (lx->timeout < lx->start_timeout)
	  lx->timeout = lx->start_timeout;
      }
      lx->in_receive = 0;
      return -1;
    case LX_SEND_DONE:
      for (i = 0; i < message_length; i++)
	lx_free_send (lx, (lx_buffer_t*) contexts [i]);
      break;
  
    case LX_PUNT:
      if (lx->duration)
	lx_clear_alarm (lx);
      lx->in_receive = 0;
      return early ? -1 : 0;
    case LX_ABORT:
      /*mi_c (("aborted with %d", length));*/
      insist (0);
      break;
    default:
      lx->in_receive = 0;
      insist (0);
  }
  lx->in_receive = 0;
  except: return 0;
}

int
lx_send_scout (lx_t*lx, int iport, lx_route_t*route, unsigned timestamp, int reason, int*oiport, int*port, int*host_type, unsigned char mac_address [6], unsigned*map_version, unsigned char map_address [6], int*level, int*flags, int*context)
{
  lx_scout_message_t*m;
  lx_scout_reply_message_t t;
  lx_small_t*p;
  int r, iiport;
  lx_map_t*map = reason == LX_SCOUTING ? &lx->maps [iport] : lx_get_map (lx, iport);
  
  insist (route);
  insist (port && oiport && host_type && mac_address && map_version && map_address && level && flags && context);
  insist (sizeof (m) <= LX_SMALL_MTU);
  insist (iport >= 0 && iport < LX_HOST_SIZE);
  insist (map);
  
  while (!(p = (lx_small_t*) lx_alloc_send (lx, LX_SMALL_SEND)))
  {
    if ((r = lx_receive (lx, &iiport, (char*) &t, sizeof (t), 0)) < 0)
      return 0;
    else if (iiport == iport && r >= (int) sizeof (t))
    {
      if (t.header.subtype == mi_htons (LX_SCOUT_REPLY) &&
	  t.header.version == LX_VERSION &&
	  !lx_memcmp (t.header.echo_address, lx->mac_address, 6))
      {
	if (lx_compare_time (lx, t.header.timestamp, mi_htonl (timestamp)))
	{
	  *port = t.header.port;
	  *host_type = t.host_type;
	  *flags = mi_htonl (t.flags);
	  *level = t.header.level;
	  *context = mi_htonl (t.header.context);
	  mi_memcpy (mac_address, t.header.sender_address, 6);
	  *map_version = mi_htonl (t.map_version);
	  mi_memcpy (map_address, t.map_address, 6);
	  *oiport = t.iport;
	  
	  return -1;
	}
      }
    }
  }

  /*mi_c (("sending scout %d to route %s", timestamp, lx_print_route (route->hops, route->length)));*/

  m = (lx_scout_message_t*) p->p;
  
  m->header.type = mi_htons (LX_TYPE);
  m->header.subtype = mi_htons (LX_SCOUT);
  m->header.timestamp = mi_htonl (timestamp);
  mi_memcpy (m->header.sender_address, lx->mac_address, 6);
  m->header.port = *port;
  m->header.level = lx->level;
  m->header.reason = reason;
  m->header.context = mi_htonl (*context);
  m->header.version = LX_VERSION;
  
  mi_memcpy (m->map_address, map->map_address, 6);
  m->map_version = mi_htonl (map->map_version);

  m->flags = 0;
  
  if (map->map_valid)
    m->flags |= LX_FLAG_MAP_VALID;
  if (map->kids_match)
    m->flags |= LX_FLAG_KIDS_MATCH;
  if (map->done_everywhere)
    m->flags |= LX_FLAG_DONE_EVERYWHERE;
  if (lx->pause)
    m->flags |= LX_FLAG_PAUSE;

  m->flags = mi_htons (m->flags);
  m->route_length = route->length;
  lx_reverse_route (m->route, route->hops, route->length);
  p->r.length = route->length;

  lx_copy_route (p->r.hops, route->hops, p->r.length);
  lx_physical_route (p->r.hops, p->r.length);
  

  mi_send (lx, iport, p->r.hops, p->r.length, (char*) m, sizeof (lx_scout_message_t), p);

  return 1;
  except: return 0;  
}

int
lx_send_scout_reply (lx_t*lx, int iport, lx_scout_message_t*scout)
{
  lx_scout_reply_message_t*m;
  lx_small_t*p;  
  lx_map_t*map = lx_get_map (lx, iport);
  
  insist (sizeof (m) <= LX_SMALL_MTU);
  insist (iport >= 0 && iport < LX_HOST_SIZE);
  insist (map);

  p = (lx_small_t*) lx_alloc_send (lx, LX_REPLY_SEND);
  insist (p);  

  m = (lx_scout_reply_message_t*) p->p;
  
  /*mi_c (("sending scout reply %d to " lx_mac_format, mi_htonl (scout->header.timestamp), lx_mac_args (scout->header.sender_address)));*/
  
  m->header.type = mi_htons (LX_TYPE);
  m->header.subtype = mi_htons (LX_SCOUT_REPLY);
  m->header.timestamp = scout->header.timestamp;
  m->header.port = scout->header.port;
  m->header.context = scout->header.context;
  m->header.level = lx->level;
  m->header.version = LX_VERSION;
  mi_memcpy (m->header.sender_address, lx->mac_address, 6);
  mi_memcpy (m->header.echo_address, scout->header.sender_address, 6);
  mi_memcpy (m->map_address, map->map_address, 6);
  m->map_version = mi_htonl (map->map_version);
  m->host_type = lx->host_type;
  m->iport = iport;
  
  m->flags = 0;  
  if (map->map_valid)
    m->flags |= LX_FLAG_MAP_VALID;
  if (map->kids_match)
    m->flags |= LX_FLAG_KIDS_MATCH;
  if (map->done_everywhere)
    m->flags |= LX_FLAG_DONE_EVERYWHERE;
  if (lx->pause)
    m->flags |= LX_FLAG_PAUSE;
  
  m->flags = mi_htonl (m->flags);  

  p->r.length = scout->route_length;  
  lx_copy_route (p->r.hops, scout->route, scout->route_length);
  lx_physical_route (p->r.hops, p->r.length);

  /*lx_dump ((char*)m, sizeof (lx_scout_reply_message_t));*/

  mi_send (lx, iport, p->r.hops, p->r.length, (char*) m, sizeof (lx_scout_reply_message_t), p);

  return 1;
  except: return 0;
}

int
lx_send_map_request (lx_t*lx, int iport, lx_route_t*route, unsigned timestamp, int type, int first)
{
  lx_map_request_message_t*m;
  lx_small_t*p;
  lx_map_t*map = &lx->maps [iport];
  
  insist (route);
  insist (sizeof (m) <= LX_SMALL_MTU);
  insist (iport >= 0 && iport < LX_HOST_SIZE);
  insist (map);
  
  mi_c (("sending map request %d to " lx_mac_format, timestamp, lx_mac_args (map->winner_address)));

  while (!(p = (lx_small_t*) lx_alloc_send (lx, LX_SMALL_SEND)))
    lx_receive (lx, 0, 0, 0, 0);  

  m = (lx_map_request_message_t*) p->p;
  
  m->header.type = mi_htons (LX_TYPE);
  m->header.subtype = mi_htons (LX_MAP_REQUEST);
  m->header.version = LX_VERSION;
  m->header.timestamp = mi_htonl (timestamp);
  mi_memcpy (m->header.sender_address, lx->mac_address, 6);
  m->type = type;
  m->first = mi_htons (first);
  
  m->route_length = route->length;
  lx_reverse_route (m->route, route->hops, route->length);

  p->r.length = route->length;
  lx_copy_route (p->r.hops, route->hops, route->length);
  lx_physical_route (p->r.hops, p->r.length);  

  mi_send (lx, iport, p->r.hops, p->r.length, (char*) m, sizeof (lx_map_request_message_t), p);

  return 1;
  except: return 0;  
}

int
lx_send_xbar (lx_t*lx, int iport, lx_route_t*route, int timestamp, int*port, int*context)
{
  lx_xbar_message_t t, *m;
  int iiport, r;
  lx_small_t*p;

  insist (route && port && context);
  insist (sizeof (m) <= LX_SMALL_MTU);
  
  while (!(p = (lx_small_t*) lx_alloc_send (lx, LX_SMALL_SEND)))
  {
    if ((r = lx_receive (lx, &iiport, (char*) &t, sizeof (t), 0)) < 0)
      return 0;
    else if (iiport == iport && r >= (int) sizeof (t))
    {
      if (t.header.subtype == mi_htons (LX_XBAR) && 
	  t.header.version == LX_VERSION &&
	  !lx_memcmp (t.header.sender_address, lx->mac_address, 6))
      {
	if (lx_compare_time (lx, t.header.timestamp, mi_htonl (timestamp)))
	{
	  *port = t.header.port;
	  *context = mi_htonl (t.header.context);
	  return -1;
	}
      }
    }
  }

  /*mi_c (("sending xbar to route %s", lx_print_route (route->hops, route->length)));*/

  m = (lx_xbar_message_t*) p->p;
  
  m->header.type = mi_htons (LX_TYPE);
  m->header.subtype = mi_htons (LX_XBAR);
  m->header.version = LX_VERSION;
  m->header.timestamp = mi_htonl (timestamp);
  mi_memcpy (m->header.sender_address, lx->mac_address, 6);
  m->header.port = *port;
  m->header.context = mi_htonl (*context);
  
  p->r.length = route->length;
  lx_copy_route (p->r.hops, route->hops, p->r.length);
  lx_physical_route (p->r.hops, p->r.length);

  mi_send (lx, iport, p->r.hops, p->r.length, (char*) m, sizeof (lx_xbar_message_t), p);
  
  return 1;
  except: return 0;
}


int
lx_wait_xbar (lx_t*lx, int iport, unsigned duration, int timestamp, int*port, int*context, int early)
{
  lx_xbar_message_t m;
  int iiport, r;

  insist (port && context);

  if (duration)
    lx_set_alarm (lx, duration);

  while ((r = lx_receive (lx, &iiport, (char*) &m, sizeof (m), early)) >= 0)
  {
    if (iiport != iport || r < (int) sizeof (m) || m.header.version != LX_VERSION || m.header.subtype != mi_htons (LX_XBAR) || lx_memcmp (m.header.sender_address, lx->mac_address, 6))
      continue;
    
    if (lx_compare_time (lx, mi_htonl (m.header.timestamp), timestamp))
      break;
  }
  
  if (r < 0)
    return 0;

  if (duration)
    lx_clear_alarm (lx);

  *port = m.header.port;
  *context = mi_htonl (m.header.context);

  return 1;
  except: return 0;
}

/*
  send a x32 probe, if there's an asynch reply while waiting for a
  send buffer, return contents of that reply.  expects normal there and
  back again route, replacement of middle routing byte done in the
  packet before it goes out.

  port is the port we're exploring on some xbar.
  
  xbar32 message is non-standard because it's constrained by the hardware
  and doesnt have a full mapper header.

*/

int lx_send_xbar32 (lx_t*lx, int iport, lx_route_t*route, int timestamp, int*port, int*context, int*id, int*mask, int*absolute_port, int*quadrant_enabled)
{
  lx_xbar32_message_t t, *m;
  int iiport, r, middle;
  lx_small_t*p;

  insist (route && port && context && absolute_port && quadrant_enabled && mask && id);
  insist (sizeof (m) <= LX_SMALL_MTU);
  
  while (!(p = (lx_small_t*) lx_alloc_send (lx, LX_SMALL_SEND)))
  {
    if ((r = lx_receive (lx, &iiport, (char*) &t, sizeof (t), 0)) < 0)
      return 0;
    else if (iiport == iport && r >= (int) sizeof (t))
    {
      if (t.subtype == mi_htons (LX_XBAR32) && t.version == LX_VERSION && !lx_memcmp (t.sender_address, lx->mac_address, 6))
      {
	if (lx_compare_time (lx, t.timestamp, mi_htonl (timestamp)))
	{
	  *port = t.port;
	  *context = mi_htonl (t.context);
	  *absolute_port = t.insert.absolute_port;
	  *quadrant_enabled = t.insert.quadrant_enabled;
	  *mask = lx_byte_swap_long (t.insert.status_mask);
	  *id = lx_byte_swap_long (t.insert.id);
	  return -1;
	}
      }
    }
  }

  /*mi_c (("sending xbar to route %s", lx_print_route (route->hops, route->length)));*/

  m = (lx_xbar32_message_t*) p->p;
  
  m->type = mi_htons (LX_TYPE);
  m->subtype = mi_htons (LX_XBAR32);
  m->version = LX_VERSION;
  m->timestamp = mi_htonl (timestamp);
  mi_memcpy (m->sender_address, lx->mac_address, 6);
  m->port = *port;
  m->context = mi_htonl (*context);

  /*zero the bytes to be replaced by the xbar32 hardware so the crc32 ends up right*/
  lx_bzero ((char*) &m->insert, sizeof (m->insert));
  
  p->r.length = route->length;
  lx_copy_route (p->r.hops, route->hops, p->r.length);
  lx_physical_route (p->r.hops, p->r.length);

  /*replace middle routing byte with special head byte of -32*/

  insist (route->length);
  middle = (route->length - 1) / 2;
  p->r.hops [middle] = 0xa0;

  mi_send (lx, iport, p->r.hops, p->r.length, (char*) m, sizeof (lx_xbar32_message_t), p);
  
  return 1;
  except: return 0;
}


int lx_wait_xbar32 (lx_t*lx, int iport, unsigned duration, int timestamp, int*port, int*context, int*id, int*mask, int*absolute_port, int*quadrant_enabled, int early)
{
  lx_xbar32_message_t m;
  int iiport, r;

  insist (port && context && absolute_port && quadrant_enabled && mask && id);

  if (duration)
    lx_set_alarm (lx, duration);

  while ((r = lx_receive (lx, &iiport, (char*) &m, sizeof (m), early)) >= 0)
  {
    if (iiport != iport || r < (int) sizeof (m) || m.version != LX_VERSION || m.subtype != mi_htons (LX_XBAR32) || lx_memcmp (m.sender_address, lx->mac_address, 6))
      continue;
    
    if (lx_compare_time (lx, mi_htonl (m.timestamp), timestamp))
      break;
  }
  
  if (r < 0)
    return 0;

  if (duration)
    lx_clear_alarm (lx);

  *port = m.port;
  *context = mi_htonl (m.context);
  *absolute_port = m.insert.absolute_port;
  *quadrant_enabled = m.insert.quadrant_enabled;
  *mask = lx_byte_swap_long (m.insert.status_mask);
  *id = lx_byte_swap_long (m.insert.id);

  return 1;
  except: return 0;
}

/*lx_wait_map_reply has a bug in it, it ignores "type". we
  depend on this bug now in order to be able to wait for LX_MAP_XBAR32
  as well as LX_MAP_XBAR*/

int
lx_wait_map_reply (lx_t*lx, int iport, unsigned duration, int timestamp, int type, lx_map_reply_message_t*reply)
{
  int iiport, r;
  lx_map_reply_message_t*m;
  
  insist (reply);

  m = (lx_map_reply_message_t*) lx->receive_buffer;

  lx_set_alarm (lx, duration);
  insist (sizeof (lx_old_map_reply_message_t) <= sizeof (lx_map_reply_message_t));
  
  while ((r = lx_receive (lx, &iiport, (char*) m, sizeof (lx_old_map_reply_message_t), 0)) >= 0)
  {
    if  (iiport != iport ||
	 m->header.version != LX_VERSION ||
	 r < (int) sizeof (lx_old_map_reply_message_t) ||
	 m->header.subtype != mi_htons (LX_MAP_REPLY) ||
	 lx_memcmp (m->header.echo_address, lx->mac_address, 6))
      continue;
    
    if (lx_compare_time (lx, mi_htonl (m->header.timestamp), timestamp))
      break; 
  }
  
  if (r < 0)
    return 0;
  lx_clear_alarm (lx);

  mi_memcpy ((char*) reply, (char*) m, r);
  
  return r;
  except: return 0;
}

int
lx_wait_scout_reply (lx_t*lx, int iport, unsigned duration, int timestamp, int*oiport, int*port, int*host_type, unsigned char mac_address [6], unsigned*map_version, unsigned char map_address [6], int*level, int*flags, int*context, int early)
{
  lx_scout_reply_message_t m;
  int iiport, r;
  
  insist (oiport && port && host_type && mac_address && map_version && map_address && flags && context);
  insist (level);
  
  if (duration)
    lx_set_alarm (lx, duration);

  while ((r = lx_receive (lx, &iiport, (char*) &m, sizeof (m), early)) >= 0)
  {
    if (iiport != iport || r < (int) sizeof (m) || m.header.version != LX_VERSION || m.header.subtype != mi_htons (LX_SCOUT_REPLY) || lx_memcmp (m.header.echo_address, lx->mac_address, 6))
      continue;

    if (lx_compare_time (lx, mi_htonl (m.header.timestamp), timestamp))
      break;
  }
  
  if (r < 0)
    return 0;
	  
  if (duration)
    lx_clear_alarm (lx);

  /*lx_dump ((char*)&m, r);*/

  *port = m.header.port;
  *host_type = m.host_type;
  mi_memcpy (mac_address, m.header.sender_address, 6);
  *map_version = mi_htonl (m.map_version);
  *level = m.header.level;
  *flags = mi_htonl (m.flags);
  *context = mi_htonl (m.header.context);
  *oiport = m.iport;
  
  mi_memcpy (map_address, m.map_address, 6);

  return 1;
  except: return 0;
}

int
lx_wait (lx_t*lx, unsigned duration, int*variable, int early)
{
  int r = 0;  
  insist (lx);
  insist (duration || early);
  
  if (variable && *variable) return 1;
  
  if (duration)
    lx_set_alarm (lx, duration);

  while ((!variable || !*variable) && (r = lx_receive (lx, 0, 0, 0, early)) >= 0)
    if (lx->die)
      return 0;

  if (r < 0)
    return 0;

  if (duration)
    lx_clear_alarm (lx);
  
  return 1;
  except: return 0;
}

void
lx_set_alarm (lx_t*lx, unsigned duration)
{
  insist (!lx->duration && duration);
  lx->set_alarm = 1;
  lx->duration = duration;  
  except:;
}

void
lx_clear_alarm (lx_t*lx)
{
  insist (lx->duration);
  insist (!lx->set_alarm);
  
  mi_set_alarm (lx, lx->duration = 0);
  except:;
}


int
lx_send_map_reply (lx_t*lx, int iport, lx_map_t*m, lx_map_request_message_t*request)
{
  int i, first;
  lx_map_reply_message_t*mr;
  lx_big_t*p;
  int piece_size;
  
  insist (m && request);
  insist (m->map_version);
  insist (m == lx_get_map (lx, iport));
  insist (m->map_valid);
  insist (iport >= 0 && iport < LX_HOST_SIZE);

  p = (lx_big_t*) lx_alloc_send (lx, LX_BIG_SEND);
  insist (p);
  
  mr = (lx_map_reply_message_t*) p->p;
  
  if (!m->routing && !lx_bsearch_host (m, request->header.sender_address, request->header.level))
  {
    mi_c ((lx_mac_format " is not in my map", lx_mac_args (request->header.sender_address)));
    mi_c (("Therefore my map is invalid."));
    lx_set_change (lx, iport, 0, 0, LX_REQUEST_NOT_IN_MAP, m->map_version, m->map_address, lx->mac_address, request->header.sender_address, 0, 0, 0);
    m->map_valid = 0;
    lx->aborted = 1;
  }

  mi_c (("sending map reply to " lx_mac_format, lx_mac_args (request->header.sender_address)));
  
  mr->header.type = mi_htons (LX_TYPE);
  mr->header.subtype = mi_htons (LX_MAP_REPLY);
  mr->header.timestamp = request->header.timestamp;
  mr->header.port = request->header.port;
  mr->header.version = LX_VERSION;
  mr->header.context = mi_htonl (m->seek_size); /*overloaded*/
  mi_memcpy (mr->header.sender_address, lx->mac_address, 6);
  mi_memcpy (mr->header.echo_address, request->header.sender_address, 6);
  mi_memcpy (mr->map_address, m->map_address, 6);

  mr->type = m->has_x32s ? LX_MAP_XBAR32 : request->type;
  mr->num_hosts = mi_htons (m->num_hosts);
  mr->num_xbars = mi_htons (m->num_xbars);
  mr->num_nodes = 0;
  mr->first = request->first;
  lx_bzero ((char*) mr->map_address, 6);
  mr->map_version = mi_htonl (m->map_version);
  mr->routing_flag = m->routing_flag;
  mr->xbar_verify_flag = m->xbar_verify_flag;
  
  first = mi_htons (request->first);

  mi_memcpy (mr->map_address, m->map_address, 6);
  
  switch (request->type)
  {
    case LX_MAP_HOST:
      piece_size = sizeof (lx_map_host_t);
      for (i = first; i < m->num_hosts && i - first < MI_MAX_HOSTS_PER_MAP_REPLY; i++)
      {
	insist (i >= 0 && i < m->num_hosts);
	lx_host2net (m, (lx_host_t*) mi_index2node (m->host_array [i]), &mr->nodes.host [i - first]);
	mr->num_nodes++;
      }
      break;
    case LX_MAP_XBAR:
      if (m->has_x32s)
      {
	piece_size = sizeof (lx_map_xbar_t);
	for (i = first; i < m->num_xbars && i - first < lx_piece_count (lx_map_xbar_t); i++)
	{
	  insist (i >= 0 && i < m->num_xbars);
	  lx_xbar2net (m, (lx_xbar_t*) mi_index2node (m->xbar_array [i]), &mr->nodes.xbar [i - first]);
	  mr->num_nodes++;
	}
      }
      else
      {
	piece_size = sizeof (lx_map_xbar16_t);
	for (i = first; i < m->num_xbars && i - first < lx_piece_count (lx_map_xbar16_t); i++)
	{
	  insist (i >= 0 && i < m->num_xbars);
	  lx_xbar162net (m, (lx_xbar_t*) mi_index2node (m->xbar_array [i]), &mr->nodes.xbar16 [i - first]);

	  mr->num_nodes++;
	}
      }
      
      break;
    default:
      insist (0);
      return 0;
  }

  p->r.length = request->route_length;
  lx_copy_route (p->r.hops, request->route, p->r.length);
  lx_physical_route (p->r.hops, p->r.length);

  piece_size = sizeof (lx_map_reply_message_t) + piece_size * ((mr->num_nodes - (mr->num_nodes ? 1 : 0)));
  insist (piece_size <= LX_BIG_MTU);
  
  mr->num_nodes = mi_htons (mr->num_nodes);
  mi_send (lx, iport, p->r.hops, p->r.length, (char*) mr, piece_size, p);
  return 1;
  except: return 0;
}


void
lx_reclaim_staged (lx_t*lx)
{
  lx_node_t*n;

  insist (lx);

  while ((n = (lx_node_t*) lx_get (&lx->staged_hosts)))
  {
    mi_punt (lx);
    lx_put (&lx->free_hosts, n);
  }

  while ((n = (lx_node_t*) lx_get (&lx->staged_xbars)))
  {
    mi_punt (lx);
    lx_put (&lx->free_xbars, n);
  }

  except:;
}	  

int lxd_cur;
int lxd_lines [LXD_MAX_LINES];
void
lxd_line (int line)
{
  if (++lxd_cur == LXD_MAX_LINES)
    lxd_cur = 0;
    
  lxd_lines [lxd_cur] = line;

}

void
lx_dump (char*p, int length)
{
#define MAX_DUMP 200
  char buffer [MAX_DUMP * 3];
  char*s = buffer;
  int i;
  
  if (length > MAX_DUMP ) length = MAX_DUMP;
  
  for (i = 0; i < length; i++)
    s += sprintf
      (s, "%2.2x%s", ((unsigned char*)p) [i],
       i  && (i + 1) % 16 == 0 ? "\n" : i && i % 2 == 1  ? " " : "");
  mi_c (("%d:\n%s\n\n", length, buffer));
}

int
lx_num_debug_hosts (lx_map_t*m)
{
  int i, c;
  lx_node_t*n;
  
  insist (m);
    
  for (c = i = 0; i < m->num_hosts; i++)
  {
    n = mi_index2node (m->host_array [i]);
    insist (n && !n->xbar);
    if (lx_host_c (n)->host_type == LX_DEBUG_TYPE)
      c++;
    mi_punt (m->lx);
  }
  return c;
  except: return 0;
}

int lx_inflate_map (lx_map_t*m, int iport, int xbar_fan, int extra_hosts)
{
  int p, i, j, num_xbars, num_host_kids, num_xbar_kids;
  unsigned char mac_address [6];
  int hosts_made = 0;
  lx_node_t*n, *nn;
  lx_route_t route;
  lx_t*lx = m->lx;
  
  insist (m && lx);
  insist (xbar_fan < LX_XBAR_SIZE);
  insist (!lx->staged_xbars.count);
  insist (iport >= 0 && iport < LX_HOST_SIZE);

  num_xbar_kids = xbar_fan - 1;
  num_host_kids = LX_XBAR_SIZE - num_xbar_kids - 1;
  insist (num_xbar_kids > 0 && num_host_kids > 0);
  
  num_xbars = 1 + extra_hosts / num_host_kids;
  insist (num_xbars > 0);
  
  i = 0;
  for (n = (lx_node_t*) lx_peek (&m->xbars); n; n = mi_index2node (n->next))
  {
    for (i = 0; i < LX_XBAR_SIZE && lx_get_node (n, n->first + i); i++);
  
    if (i < LX_XBAR_SIZE)
      break;
  }

  if (!n)
    return 0;
  p = n->first + i;
  insist (p > -LX_XBAR_SIZE && p < LX_XBAR_SIZE);

  nn = lx_new_xbar (m, iport, lx_append_route (&route, &lx_xbar_c (n)->routes [iport], p), 0, 0, 0);
  insist (nn);
  lx_connect (n, p, nn, 0);
  lx_put (&lx->staged_xbars, nn);

  mac_address [0] = 0;
  mac_address [1] = 0x60;
  mac_address [2] = 0xdd;
  
  while ((n = lx_get (&lx->staged_xbars)))
  {
    lx_put (&m->xbars, n);
    
    for (i = 0, p = 1; i < num_xbar_kids && num_xbars; i++, num_xbars--, p++)
    {
      nn = lx_new_xbar (m, iport, lx_append_route (&route, &lx_xbar_c (n)->routes [iport], p), 0, 0, 0);
      insist (nn);
      lx_connect (n, p, nn, 0);
      lx_put (&lx->staged_xbars, nn);
    }
    for (i = 0; i < num_host_kids && hosts_made < extra_hosts; i++, p++)
    {
      insist (p < LX_XBAR_SIZE);

      hosts_made++;
      
      for (j = 3; j < 6; j++)
	mac_address [j] = (extra_hosts + 1 - hosts_made) >> (8 * (5 - j));
      
      nn = lx_new_host (m, iport, 0, mac_address, LX_DEBUG_TYPE, 0, lx_append_route (&route, &lx_xbar_c (n)->routes [iport], p));
      insist (nn);
      lx_connect (n, p, nn, 0);
      insist (m->hosts.tail);
      insist (lx_hostcmp (lx_host_c (m->hosts.tail)->mac_address, lx_host_c (m->hosts.tail)->level, mac_address, 0) > 0);
      lx_put (&m->hosts, nn);
    }
  }
  return 1;
  except: return 0;
}


void lx_xbar2net (lx_map_t*m, lx_xbar_t*from, lx_map_xbar_t*to)
{
  int i;
  lx_node_t*n;
  
  insist (m && from && to);
  
  to->level = from->level;
  to->id_low = mi_htons ((from->id & 0xffff));
  to->id_high = mi_htons ((from->id >> 16));
  
  for (i = 0; i < LX_XBAR_SIZE; i++)
  {
    if ((n = mi_index2node (lx_node_c (from)->links [i].index)))
    {
      to->links [i].index = 1 + (n->index + (n->xbar ? m->num_hosts : 0));
      insist (to->links [i].index > 0 && to->links [i].index <= m->num_hosts + m->num_xbars);
      to->links [i].port = lx_node_c (from)->links [i].port -  n->first;
      to->links [i].index = mi_htons (to->links [i].index);
    }
    else
      to->links [i].index = to->links [i].port = 0;
  }
  
  except:;
}

void lx_xbar162net (lx_map_t*m, lx_xbar_t*from, lx_map_xbar16_t*to)
{
  int i;
  lx_node_t*n;
  
  insist (m && from && to);
  
  to->level = from->level;
  
  for (i = 0; i < LX_XBAR16_SIZE; i++)
  {
    if ((n = mi_index2node (lx_node_c (from)->links [i].index)))
    {
      to->links [i].index = 1 + (n->index + (n->xbar ? m->num_hosts : 0));
      insist (to->links [i].index > 0 && to->links [i].index <= m->num_hosts + m->num_xbars);
      to->links [i].port = lx_node_c (from)->links [i].port -  n->first;
      to->links [i].index = mi_htons (to->links [i].index);
    }
    else
      to->links [i].index = to->links [i].port = 0;
  }
  
  except:;
}

void lx_net2xbar (lx_map_t*m, lx_map_xbar_t*from, lx_xbar_t*to, int num_hosts, int num_xbars)
{
  int i;
  lx_node_t*n;
  
  insist (m && from && to);
  insist (!m->map_valid);

  lx_node_c (to)->first = 0;
  to->level = from->level;
  
  for (i = 0; i < LX_XBAR_SIZE; i++)
  {
    int index = mi_htons (from->links [i].index);
    
    insist (index >= 0 && index <= num_hosts + num_xbars);

    if (index)
    {
      index--;
      n = mi_index2node ((index < num_hosts ? m->host_array [index] : m->xbar_array [index - num_hosts]));
      insist (n);
      
      lx_node_c (to)->links [i].index = mi_node2index (n);
      lx_node_c (to)->links [i].port = from->links [i].port;
    }
    else
      lx_node_c (to)->links [i].index = 0;
  }
  to->id = (mi_htons (from->id_low) & 0xffff) | (mi_htons (from->id_high) << 16);
  
  except:;
}

void lx_net2xbar16 (lx_map_t*m, lx_map_xbar16_t*from, lx_xbar_t*to, int num_hosts, int num_xbars)
{
  int i;
  lx_node_t*n;
  
  insist (m && from && to);
  insist (!m->map_valid);

  lx_node_c (to)->first = 0;
  to->level = from->level;
  
  for (i = 0; i < LX_XBAR16_SIZE; i++)
  {
    int index = mi_htons (from->links [i].index);
    
    insist (index >= 0 && index <= num_hosts + num_xbars);

    if (index)
    {
      index--;
      n = mi_index2node ((index < num_hosts ? m->host_array [index] : m->xbar_array [index - num_hosts]));
      insist (n);
      
      lx_node_c (to)->links [i].index = mi_node2index (n);
      lx_node_c (to)->links [i].port = from->links [i].port;
    }
    else
      lx_node_c (to)->links [i].index = 0;
  }
  to->id = 0;
  except:;
}

void
lx_host2net (lx_map_t*m, lx_host_t*from, lx_map_host_t*to)
{
  lx_node_t*n;
  int i;
  
  insist (m && from && to);
  insist (!lx_zero (from->mac_address, 6));
  
  mi_memcpy (to->mac_address, from->mac_address, 6);
  to->host_type = from->host_type;
  to->level = from->level;
  
  for (i = 0; i < LX_HOST_SIZE; i++)
  {
    if ((n = mi_index2node (lx_node_c (from)->links [i].index)))
    {  
      to->links [i].index = 1 + (n->index + (n->xbar ? m->num_hosts : 0));
      insist (to->links [i].index > 0 && to->links [i].index <= m->num_hosts + m->num_xbars);
      to->links [i].port = lx_node_c (from)->links [i].port - n->first;
      to->links [i].index = mi_htons (to->links [i].index);
    }
    else
      to->links [i].index = to->links [i].port = 0;
  }
  
  except:;
}

void
lx_net2host (lx_map_t*m, lx_map_host_t*from, lx_host_t*to, int num_hosts, int num_xbars)
{ 
  int i, index;
  lx_node_t*n;
  
  insist (m && from && to);
  insist (num_hosts <= LX_MAX_HOSTS);
  insist (num_xbars <= LX_MAX_XBARS);
  insist (!m->map_valid);
  
  lx_node_c (to)->first = 0;
  insist (!lx_zero (from->mac_address, 6));
  
  mi_memcpy (to->mac_address, from->mac_address, 6);
  to->host_type = from->host_type;
  to->level = from->level;
  
  for (i = 0; i < LX_HOST_SIZE; i++)
  {
    index = mi_htons (from->links [i].index);
    insist (index >= 0 && index <= num_hosts + num_xbars);

    if (index)
    {
      index--;
      n = mi_index2node ((index < num_hosts ? m->host_array [index] : m->xbar_array [index - num_hosts]));
      insist (n);
      
      lx_node_c (to)->links [i].index = mi_node2index (n);
      lx_node_c (to)->links [i].port = from->links [i].port;
    }
    else
      lx_node_c (to)->links [i].index = 0;
  }
  
  except:;
}

int
lx_set_routes (lx_map_t*m, int iport, int index)
{
  int i, j;
  lx_node_t*n;
  
  insist (m);
  insist (iport >= 0 && iport < MI_MY_SIZE);

  insist (m->root);
  if (!lx_get_node (m->root, iport))
    return 1;
  
  for (i = 0; i < m->num_hosts; i++)
  {
    n = mi_index2node (m->host_array [i]);
    insist (n && !n->xbar);
    
    for (j = 0; j < LX_HOST_SIZE; j++)
    {
      if (lx_get_node (n, j))
      {
	{
#ifndef MI_XM

	  lx_node_t *fn;
	  int last_port;
	  
	  fn = lx_follow (m->root, iport, &lx_host_c (n)->routes [iport][j],
	             &last_port);
	  if (!m->num_xbars && (fn != n || last_port != j))
	    continue;
	  insist (fn == n);
	  insist (last_port == j);
#endif
	}

	mi_set_route (m->lx, index,
	              lx_host_c (n)->mac_address,
	              lx_host_c (n)->host_type,
		      n->index, iport, j, 
		      lx_host_c (n)->routes [iport] [j].hops, 
		      lx_host_c (n)->routes [iport] [j].length);
      }
      mi_punt (m->lx);
    }
  }  

  return 1;
  
  except: return 0;
}

lx_map_t *
lx_get_map (lx_t*lx, int iport)
{
  int i;
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);

  for (i = 0; i < MI_MY_SIZE; i++)
  {
    if (lx->maps[i].map_valid && lx_get_node (lx->maps [i].root, iport))
      return &lx->maps [i];
  }
  return &lx->maps [iport];

  except: return 0;
}


unsigned char *
lx_get_first_host (lx_t*lx, int iport, int host_type)
{
  lx_map_t*m;
  lx_host_t*h;
  
  insist (lx);
  insist (iport >= 0 && iport <= MI_MY_SIZE);
  
  if (!(m = lx_get_map (lx, iport)))
    return 0;

  lx->first_host = 0;
  lx->first_host_type = host_type;
  lx->first_host_map = m;
  
  for (lx->first_host = 0; lx->first_host < m->num_hosts; lx->first_host++)
  {
    h = lx_host_c (mi_index2node (m->host_array [lx->first_host]));
    insist (h);
    if (h->host_type == host_type)
      return h->mac_address;
    mi_punt (lx);
  }

  except: return 0;
}

unsigned char *
lx_get_next_host (lx_t*lx)
{
  lx_host_t*h;
  
  insist (lx && lx->first_host_map);
  
  for (++lx->first_host; lx->first_host < lx->first_host_map->num_hosts; lx->first_host++)
  {
    h = lx_host_c (mi_index2node (lx->first_host_map->host_array [lx->first_host]));
    insist (h);
    if (h->host_type == lx->first_host_type)
      return h->mac_address;
    mi_punt (lx);
  }

  except: return 0;
}

int
lx_send_change_report (lx_t*lx, int iport, lx_change_t*change, unsigned char mac_address [6], lx_route_t*route, unsigned timestamp)
{
  lx_big_change_report_message_t*m;
  lx_scout_reply_message_t t;
  lx_small_t*p;
  int r, iiport;

  insist (lx && change && route && mac_address);
  insist (sizeof (*m) <= LX_BIG_MTU);
  insist (iport >= 0 && iport < MI_MY_SIZE);

  while (!(p = (lx_small_t*) lx_alloc_send (lx, LX_BIG_SEND)))
  {
    if ((r = lx_receive (lx, &iiport, (char*) &t, sizeof (t), 0)) < 0)
      return 0;
    else if (iiport == iport && r >= (int) sizeof (t))
    {
      if (t.header.subtype == mi_htons (LX_SCOUT_REPLY) &&
	  t.header.version == LX_VERSION &&
	  !lx_memcmp (t.header.echo_address, lx->mac_address, 6) &&
	  lx_compare_time (lx, t.header.timestamp, mi_htonl (timestamp)))
	return -1;
    }
  }

  m = (lx_big_change_report_message_t*) p->p;
  
  m->change.header.type = mi_htons (LX_TYPE);
  m->change.header.subtype = change->xid ? mi_htons (LX_BIG_CHANGE_REPORT) : mi_htons (LX_CHANGE_REPORT);
  m->change.header.timestamp = mi_htonl (timestamp);
  m->change.header.level = lx->level;
  m->change.header.version = LX_VERSION;

  mi_memcpy (m->change.header.sender_address, lx->mac_address, 6);
  mi_memcpy (m->change.header.echo_address, change->leaf_address, 6);
  mi_memcpy (m->change.map_address, change->map_address, 6);
  mi_memcpy (m->change.mac_address, change->mac_address, 6);
  m->change.node = mi_htons (change->node);
  m->change.change = change->change;
  m->change.port = change->port;
  m->xid = mi_htonl (change->xid);

  m->change.route_length = route->length;
  lx_reverse_route (m->change.route, route->hops, route->length);

  p->r.length = route->length;
  lx_copy_route (p->r.hops, route->hops, route->length);
  lx_physical_route (p->r.hops, p->r.length);

  mi_c (("sending change report %d to " lx_mac_format, timestamp, lx_mac_args (mac_address)));

  mi_send (lx, iport, p->r.hops, p->r.length, (char*) m, change->xid ? sizeof (lx_big_change_report_message_t) : sizeof (lx_change_report_message_t), p);

  return 1;
  except: return 0;
}

int
lx_send_change_report_reply (lx_t*lx, int iport, lx_change_report_message_t*report)
{
  lx_scout_reply_message_t*m;
  lx_small_t*p;  
  lx_map_t*map = lx_get_map (lx, iport);
  
  insist (sizeof (m) <= LX_SMALL_MTU);
  insist (iport >= 0 && iport < LX_HOST_SIZE);
  insist (map);
  
  p = (lx_small_t*) lx_alloc_send (lx, LX_REPLY_SEND);
  insist (p);  

  m = (lx_scout_reply_message_t*) p->p;
  
  /*mi_c (("sending change report reply %d to " lx_mac_format, mi_htonl (scout->header.timestamp), lx_mac_args (scout->header.sender_address)));*/
  
  m->header.type = mi_htons (LX_TYPE);
  m->header.subtype = mi_htons (LX_SCOUT_REPLY);
  m->header.timestamp = report->header.timestamp;
  m->header.port = 0;
  m->header.context = 0;
  m->header.level = lx->level;
  m->header.version = LX_VERSION;
  mi_memcpy (m->header.sender_address, lx->mac_address, 6);
  mi_memcpy (m->header.echo_address, report->header.sender_address, 6);
  mi_memcpy (m->map_address, map->map_address, 6);
  m->map_version = mi_htonl (map->map_version);
  m->host_type = lx->host_type;
  m->iport = iport;
  
  m->flags = 0;  
  if (map->map_valid)
    m->flags |= LX_FLAG_MAP_VALID;
  if (map->kids_match)
    m->flags |= LX_FLAG_KIDS_MATCH;
  if (map->done_everywhere)
    m->flags |= LX_FLAG_DONE_EVERYWHERE;
  if (lx->pause)
    m->flags |= LX_FLAG_PAUSE;
  
  m->flags = mi_htonl (m->flags);
  
  p->r.length = report->route_length;  
  lx_copy_route (p->r.hops, report->route, report->route_length);
  lx_physical_route (p->r.hops, p->r.length);

  /*lx_dump ((char*)m, sizeof (lx_scout_reply_message_t));*/

  mi_send (lx, iport, p->r.hops, p->r.length, (char*) m, sizeof (lx_scout_reply_message_t), p);

  return 1;
  except: return 0;
}


int
lx_0011 (lx_t*lx, int n)
{
  lx_map_t*m;
  lx_node_t*o;
  int i;
  
  for (i = 0; i < LX_HOST_SIZE; i++)
  {
    m = lx_get_map (lx, i);
    insist (m);
    if (n >= m->num_hosts || !lx_get_node (m->root, i))
      return 0;

    o = mi_index2node (m->host_array [n]);
    insist (o && !o->xbar);

    if (!lx_get_node (o, i))
      return 0;
  }
  return 1;
  except: return 0;
}
